'use strict';

/* --------------------------------------------------------------
 single_checkbox.js 2016-10-17
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Single Checkbox Widget
 *
 * This widget originates from the "single-checkbox" mode of the existing checkbox widget. Because of the
 * increased complexity of the old widget code, the single-checkbox mode is now served by this file. Apply the
 * widget in a parent container and it will search and convert all the instances into fine checkboxes.
 *
 * If you want to dynamically change the state of the checkbox, apply the new "checked" prop in the input:checkbox
 * element and then trigger the "change" event. This will also update the .single-checkbox wrapper.
 *
 * ### Options
 *
 * **Selector | `data-single_checkbox-selector` | String | Optional**
 *
 * Set the selector of the checkboxes to be converted to single checkbox instances. It defaults to **input:checkbox**.
 *
 * ### Methods
 *
 * **Checked**
 *
 * ```js
 * // Set the checked value of the single checkbox selection (no change event will be triggered!).
 * $('table input:checkbox').single_checkbox('checked', true);
 * ```
 *
 * ### Events
 *
 * **Initialization**
 *
 * This module triggers the "single_checkbox:ready" event, which will be handled in the `checkbox_mapping.js` file.
 * It is needed to add the caret symbol next to the checkbox and to open the multi select dropdown menu next to it.
 *
 * ### Examples
 *
 * In the following example the checkbox element will be converted into a single-checkbox instance.
 *
 * ```html
 * <div class="wrapper" data-gx-widget="single_checkbox">
 *   <input type="checkbox" />
 * </div>
 * ```
 */
gx.widgets.module('single_checkbox', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        selector: 'input:checkbox'
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Set the "checked" property of the single checkbox instances.
     *
     * This method will update the value and display of the widgets without triggering a "change" event.
     *
     * @param {Boolean} isChecked The checkbox values will be updated along with their representation.
     *
     * @return {jQuery} Returns the jQuery selector for chained calls.
     */
    function _checked(isChecked) {
        $(this).prop('checked', isChecked);
        _onCheckboxChange.call(this);
        return $(this);
    }

    /**
     * Add Public Module Methods
     *
     * Example: $('input:checkbox').single_checkbox('checked', false);
     */
    function _addPublicMethod() {
        if ($.fn.single_checkbox) {
            return; // Method is already registered.
        }

        $.fn.extend({
            single_checkbox: function single_checkbox(action) {
                switch (action) {
                    case 'checked':
                        for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
                            args[_key - 1] = arguments[_key];
                        }

                        return _checked.apply(this, args);
                }
            }
        });
    }

    /**
     * Wrap the checkbox elements with an external <span> which will be styled with CSS.
     *
     * This method will also bind the event handlers of each checkbox element.
     */
    function _wrapCheckboxElements() {
        $this.find(options.selector).each(function () {
            var checked = $(this).prop('checked') ? 'checked' : '';
            var disabled = $(this).prop('disabled') ? 'disabled' : '';
            var title = $(this).attr('title');

            $(this).css({
                position: 'absolute',
                left: '-100000px'
            }).wrap('<span class="single-checkbox ' + checked + ' ' + disabled + '" ' + (title ? title = '"' + title + '"' : '') + '></span>').parent().append('<i class="fa fa-check"></i>');

            $(this).on('focus', _onCheckboxFocus).on('blur', _onCheckboxBlur).on('change', _onCheckboxChange);
        });
    }

    /**
     * On Checkbox Change
     *
     * This event handler will make sure that the parent has the correct classes depending the checkbox state.
     */
    function _onCheckboxChange() {
        var $wrapper = $(this).parent();
        var isChecked = $(this).prop('checked');

        if (isChecked && !$wrapper.hasClass('checked')) {
            $wrapper.addClass('checked');
        } else if (!isChecked && $wrapper.hasClass('checked')) {
            $wrapper.removeClass('checked');
        }
    }

    /**
     * On Checkbox Focus
     *
     * This event handler will add the "focused" class which is used for styling.
     */
    function _onCheckboxFocus() {
        $(this).parent().addClass('focused');
    }

    /**
     * On Checkbox Blur
     *
     * This event handler will remove the "focused" class which is used for styling.
     */
    function _onCheckboxBlur() {
        $(this).parent().removeClass('focused');
    }

    /**
     * On Wrapper Click
     *
     * This event handler will delegate the click to the checkbox and must not change the state of the widget.
     *
     * @param event {object}
     */
    function _onWrapperClick(event) {
        event.stopPropagation();

        if ($(this).hasClass('disabled') || $this.find('.dataTables_empty').length) {
            return;
        }

        var $checkbox = $(this).children('input:checkbox');

        $checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        _addPublicMethod();
        _wrapCheckboxElements();

        $this.off('click', '.single-checkbox').on('click', '.single-checkbox', _onWrapperClick);

        $this.trigger('single_checkbox:ready'); // Needed for the checkbox_mapping.js file.

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
